//+------------------------------------------------------------------+
//|                              LightGBM timeseries forecasting.mq5 |
//|                                     Copyright 2023, Omega Joctan |
//|                        https://www.mql5.com/en/users/omegajoctan |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, Omega Joctan"
#property link      "https://www.mql5.com/en/users/omegajoctan"
#property version   "1.00"

#resource "\\Files\\lightgbm.Timeseries Forecasting.D1.onnx" as uchar lightgbm_onnx[]
#include <LightGBM.mqh>
CLightGBM lgb;

#include <Trade\Trade.mqh>
#include <Trade\PositionInfo.mqh>

CTrade m_trade;
CPositionInfo m_position;
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

input ENUM_TIMEFRAMES timeframe = PERIOD_D1;
input int magic_number = 1945;
input int slippage = 50;
input int stoploss = 500;
input int takeprofit = 700;

int OldNumBars =0;
double lotsize;

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

struct ohlc_struct 
{
   vector open;
   vector high;
   vector low;
   vector close;
   
   matrix MATRIX; //this stores all the vectors all-together
   
   void AddCopyRates(string symbol, ENUM_TIMEFRAMES tf, ulong start, ulong size)
    {
      open.CopyRates(symbol, tf, COPY_RATES_OPEN, start, size); 
      high.CopyRates(symbol, tf, COPY_RATES_HIGH, start, size); 
      low.CopyRates(symbol, tf, COPY_RATES_LOW, start, size); 
      close.CopyRates(symbol, tf, COPY_RATES_CLOSE, start, size); 
      
      this.MATRIX.Resize(open.Size(), 4); //we resize it to match one of the vector since all vectors are of the same size
      
      this.MATRIX.Col(open, 0);
      this.MATRIX.Col(high, 1);
      this.MATRIX.Col(low, 2);
      this.MATRIX.Col(close, 3);
    }
};
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//---
   
   if (!lgb.Init(lightgbm_onnx)) //Initialize the LightGBM model
     return INIT_FAILED;
   
   m_trade.SetExpertMagicNumber(magic_number);
   m_trade.SetDeviationInPoints(slippage);
   m_trade.SetMarginMode();
   m_trade.SetTypeFillingBySymbol(Symbol());
           
   lotsize = SymbolInfoDouble(Symbol(), SYMBOL_VOLUME_MIN);
   
//---
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//---
    
   if (NewBar()) //Trade at the opening of a new candle
    {
     vector input_vector = input_data(); 
     long signal = lgb.predict_bin(input_vector);
     
     Print("Signal==",signal);
     
   //---
     
      MqlTick ticks;
      SymbolInfoTick(Symbol(), ticks);
      
      if (signal==1) //if the signal is bullish
       {
          if (!PosExists(POSITION_TYPE_BUY)) //There are no buy positions
           {
             if (!m_trade.Buy(lotsize, Symbol(), ticks.ask, ticks.bid-stoploss*Point(), ticks.ask+takeprofit*Point())) //Open a buy trade
               printf("Failed to open a buy position err=%d",GetLastError());
           }
       }
      else if (signal==0) //Bearish signal
        {
          if (!PosExists(POSITION_TYPE_SELL)) //There are no Sell positions
            if (!m_trade.Sell(lotsize, Symbol(), ticks.bid, ticks.ask+stoploss*Point(), ticks.bid-takeprofit*Point())) //open a sell trade
               printf("Failed to open a sell position err=%d",GetLastError());
        }
      else //There was an error
        return;
    }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector input_data()
 {
//--- getting Open, high, low and close prices
   
   matrix DATASET = {};
   
   ohlc_struct OHLC;
   vector time_vector; //we wanna add time vector 
   
   int start_bar = 1, bars = 1; //get the last closed bar info only
   
   OHLC.AddCopyRates(Symbol(), timeframe, start_bar, bars);
   time_vector.CopyRates(Symbol(), timeframe, COPY_RATES_TIME, start_bar, bars); //copy the time in seconds
   
//--- Getting the lagged values of Open, High, low and close prices

   ohlc_struct  lag_1;
   lag_1.AddCopyRates(Symbol(), timeframe, start_bar+1, bars);
   
   ohlc_struct  lag_2;
   lag_2.AddCopyRates(Symbol(), timeframe, start_bar+2, bars);
   
   ohlc_struct  lag_3;
   lag_3.AddCopyRates(Symbol(), timeframe, start_bar+3, bars);

//--- Adding the lagged features to the dataset matrix 

   DATASET = concatenate(DATASET, OHLC.MATRIX);
   DATASET = concatenate(DATASET, lag_1.MATRIX);
   DATASET = concatenate(DATASET, lag_2.MATRIX);
   DATASET = concatenate(DATASET, lag_3.MATRIX);
   
   
//---  Rolling Statistics 

   int ma_handle = iMA(Symbol(),timeframe,30,0,MODE_SMA,PRICE_WEIGHTED); //The Moving averaege for 30 days
   if (ma_handle == INVALID_HANDLE)
     {
       printf("Invalid MA handle err=%d",GetLastError());
       DebugBreak();
     }
     
   int stddev = iStdDev(Symbol(), timeframe, 7,0,MODE_SMA,PRICE_WEIGHTED); //The standard deviation for 7 days
   if (stddev == INVALID_HANDLE)
     {
       printf("Invalid stddev handle err=%d",GetLastError());
       DebugBreak();
     }
   
   vector SMA_BUFF, STDDEV_BUFF;
   
   SMA_BUFF.CopyIndicatorBuffer(ma_handle,0,start_bar, bars);
   STDDEV_BUFF.CopyIndicatorBuffer(stddev, 0, start_bar, bars);
   
   DATASET = concatenate(DATASET, SMA_BUFF);
   DATASET = concatenate(DATASET, STDDEV_BUFF);
   
//--- Datetime Features
   
   ulong size = time_vector.Size(); 
   vector DAY(size), DAYOFWEEK(size), DAYOFYEAR(size), MONTH(size);
   
   MqlDateTime time_struct;
   string time = "";
   for (ulong i=0; i<size; i++)
     {
       time = (string)datetime(time_vector[i]); //converting the data from seconds to date then to string
       TimeToStruct((datetime)StringToTime(time), time_struct); //convering the string time to date then assigning them to a structure
       
       DAY[i] = time_struct.day;
       DAYOFWEEK[i] = time_struct.day_of_week;
       DAYOFYEAR[i] = time_struct.day_of_year;
       MONTH[i] = time_struct.mon;
     }
   
   DATASET = concatenate(DATASET, DAY); //day of the month
   DATASET = concatenate(DATASET, DAYOFWEEK); //day of the week; monday, tuesday...
   DATASET = concatenate(DATASET, DAYOFYEAR); //a day out of approx 365 days in a calendar year
   DATASET = concatenate(DATASET, MONTH); //A month
   
//--- Differencing | Regular differencing lag 1
   
   vector diff_lag_1_open = OHLC.open - lag_1.open;
   vector diff_lag_1_high = OHLC.high - lag_1.high;
   vector diff_lag_1_low = OHLC.low - lag_1.low;
   vector diff_lag_1_close = OHLC.close - lag_1.close;
   
   DATASET = concatenate(DATASET, diff_lag_1_open);
   DATASET = concatenate(DATASET, diff_lag_1_high);
   DATASET = concatenate(DATASET, diff_lag_1_low);
   DATASET = concatenate(DATASET, diff_lag_1_close);

// converting 1xN matrix to array
   
   vector v(DATASET.Cols());
   for (ulong i=0; i<DATASET.Cols(); i++)
     v[i] = DATASET[0][i];
   
   return v;
 }
//+------------------------------------------------------------------+
//|                                                                  |
//|   Appends matrix mat1 to the end of mat2, setting axis=1 appends |
//|  mat2 at the end of mat1 columns while axis=0 will make the      |
//| function add the new matrix mat2 at the end of matrix mat1       |
//|                                                                  |
//+------------------------------------------------------------------+
matrix concatenate(const matrix &mat1, const matrix &mat2, int axis = 1)
 {
     matrix m_out = {};

     if ((axis == 0 && mat1.Cols() != mat2.Cols() && mat1.Cols()>0) || (axis == 1 && mat1.Rows() != mat2.Rows() && mat1.Rows()>0)) 
       {
         Print(__FUNCTION__, "Err | Dimensions mismatch for concatenation");
         //DebugBreak();
         return m_out;
       }

     if (axis == 0) {
         m_out.Resize(mat1.Rows() + mat2.Rows(), MathMax(mat1.Cols(), mat2.Cols()));

         for (ulong row = 0; row < mat1.Rows(); row++) {
             for (ulong col = 0; col < m_out.Cols(); col++) {
                 m_out[row][col] = mat1[row][col];
             }
         }

         for (ulong row = 0; row < mat2.Rows(); row++) {
             for (ulong col = 0; col < m_out.Cols(); col++) {
                 m_out[row + mat1.Rows()][col] = mat2[row][col];
             }
         }
     } else if (axis == 1) {
         m_out.Resize(MathMax(mat1.Rows(), mat2.Rows()), mat1.Cols() + mat2.Cols());

         for (ulong row = 0; row < m_out.Rows(); row++) {
             for (ulong col = 0; col < mat1.Cols(); col++) {
                 m_out[row][col] = mat1[row][col];
             }

             for (ulong col = 0; col < mat2.Cols(); col++) {
                 m_out[row][col + mat1.Cols()] = mat2[row][col];
             }
         }
     }
   return m_out;
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
matrix concatenate(const matrix &mat, const vector &v, int axis=1)
 {
   matrix ret= mat;
     
   ulong new_rows, new_cols;
   
   if (axis == 0) //place it along the rows
    {
      if (mat.Cols() == 0)
        ret.Resize(mat.Rows(), v.Size());
        
      new_rows = ret.Rows()+1; new_cols = ret.Cols();
                 
      if (v.Size() != new_cols)
        {
          Print(__FUNCTION__," Dimensions don't match the vector v needs to have the same size as the number of columns in the original matrix");
          //DebugBreak();
          return ret;
        }
      
      ret.Resize(new_rows, new_cols);
      ret.Row(v, new_rows-1);
    }
   else if (axis == 1)
     {
         if (mat.Rows() == 0)
           ret.Resize(v.Size(), ret.Cols());
           
        new_rows = ret.Rows(); new_cols = ret.Cols()+1;
        
        if (v.Size() != new_rows)
          {
            Print(__FUNCTION__," Dimensions don't match the vector v needs to have the same size as the number of rows in the original matrix");
            //DebugBreak();
            return ret;
          }
        
        ret.Resize(new_rows, new_cols);
        ret.Col(v, new_cols-1);
     }
   else 
     {
       Print(__FUNCTION__," Axis value Can either be 0 or 1");
       //DebugBreak();
       return ret;
     }

//---
   return ret;
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool NewBar()
  {
   int CurrentNumBars = Bars(Symbol(),Period());
   if(OldNumBars!=CurrentNumBars)
     {
      OldNumBars = CurrentNumBars;
      return true;
     }
   return false;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool PosExists(ENUM_POSITION_TYPE type)
 {
    for (int i=PositionsTotal()-1; i>=0; i--)
      if (m_position.SelectByIndex(i))
         if (m_position.Symbol()==Symbol() && m_position.Magic() == magic_number && m_position.PositionType()==type)
            return (true);
            
    return (false);
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

